/** King's Valley 1 Plus Custom Level read/write plugin for Tiled 
 * v1.0 by FRS, 2021. Based on:
 * - videogame-format.js by diogoeichert
 * - "How to write a custom Text File importer for Tiled Map Editor" guide, by Mezzmer & eishiya
 * 
 * License: CC:BY-SA
 * Requires: Tiled v1.7.2 or higher
 * 
 * This file was designed as an easy-to-adapt template for other 8bit projects
 * Special thanks for MsxKun@msx.org, for giving me the pointers on where to begin
 * 
 * Install this script as explained in the following article:
 * https://doc.mapeditor.org/en/stable/reference/scripting/#binaryfile
*/

/*	KV1-Plus Game file "GAMExx.KVG" format
	Obs1: EOPL = End Of the previous List
	Obs2: are fields are byte-sized unless otherwise specified
	Obs3: all x an y coordinates are in pixels, unless otherwise specified
	Obs4: All y coordinates are offset by +8, because the game discards the first line of the tile map

	offset
	===Header===
	+000h~+007h : MagicID Signature: KV1Plus\0x1A
	+008h		: Reserved for Theme
	+009h		: Reserved for Song
	+00Ah~+00Ch : Author TLA
	+00Dh~+01Fh : Title

	+020h~+05Fh	: Empty line with zeroes
	===Map Data===
	+060h~+89Fh	: 96x22 tile map
	+8A0h~+8FFh	: Empty line with FFh
	===Tail===
	+900h		: Exit Door Array[4]
					[0]=Exit Up
					[1]=Exit Down
					[2]=Exit Left
					[3]=Exit Right
	 > Exit Door object. size=5 bytes
	 +00h		: y
	 +01h		: x
	 +02h		: room
	 +03h		: Level number where this door leads to, when exited through here
	 +04h		: Arrival door to connect: 1=up, 2=down, 4=left, 8=right. The door *must* exist on the destination pyramid, or the game will crash.

	+914h		: Number of mummies: ***Defines the list size too
	+915h		: Mummy list (up to ? objects)
	 > Mummy object. size=4 bytes 
	   +00h		: y
	   +01h		: x
	   +02h		: room
	   +03h		: Mummy: type {0 ~ 4}

	EOPL+1		: Number of Orbs
	EOPL+2		: Orb list (up to ? objects)
	 > Orb object. size=4 bytes
	   +00h		: Type {31h,41h,51h,61h,71h,81h}
	   +01h		: y
	   +02h		: x
  	   +03h		: room

	EOPL+1		: Number of Swords
	EOPL+2		: Sword list (up to ? objects)
	 > Sword object. size=3 bytes
	   +00h		: y
	   +01h		: x
	   +02h		: room

	EOPL+1		: Number of Pickaxes
	EOPL+2		: Pickaxe list (up to ? objects)
	 > Pickaxe object. size=3 bytes
	   +00h		: y
	   +01h		: x
	   +02h		: room

	EOPL+1		: Number of flip-Doors
	EOPL+2		: flip-Doors list (up to ? objects)
	 > flip-Door object. size=5 bytes
	  +00h		: (height-2)<<1. Obs: 1 < height < 5
	  			  bit0=1: autoflip the door side moments after the game starts
	  +01h		: y
	  +02h		: x
	  +03h		: room
	  +04h		: flags	{ 00=defective 1way, 04=RL, 08=LR, 12=? }

	EOPL+1		: Number of trap-Walls
	EOPL+2		: trap-Walls list (up to ? objects)
	 > Trap-Wall object. size=3 bytes
	   +00h		: y		(must be at the height of the player head)
	   +01h		: x
	   +02h		: room

	EOPL+1		: 00h,00h,01h,FFh: EOF signature
	EOPL+2		: Padding with non-specific data until +A03h
	


*/

let objIdx=0x100;								// Impossible tile numbers are used for objects
// ==== Auxiliary structures and functions for this file format ====
var kv1gam = {
	// === File-format Configuration  ===
	displayName		: "KV1PlusGame",			// Name of this file format to be displayed on Tiled
	description		: "King's Valley 1 Plus Game",	// Description of this file format to be displayed on Tiled
	fileExtension	: "KVG",					// File extension
	magicID			: "KV1Plus\x1A",			// Unique Magic ID on the file header
	mapWidths		: [32,64,96],				// Allowed map widhts
	mapHeights		: [22],						// Allowed map heights
	headerLength	: 0x60,						// Header length
	tailLength		: 0x104,					// Tail lenght
	mapLength		: 32*3*22+32*3,				// Map full lenght
	maxTile			: 128,						// Highest allowed tile number
	tileSetName		: "KV1TileSet.tsx",			// Tileset to use with this format. Must be present on the Tiled Extensions folder.
	debugVerbose	: true,					// Enable the verbose logging for debug
	layerNames: { 								// Name of each layer used on Tiled
		MAINMAP: "Tile Map"
	},
	attrNames: {				// Name of each attribute on the Tiled 'Header Object layer'
		THEME			: "Theme",
		SONG			: "Song",
		AUTHOR			: "Author",
		TITLE			: "Title",
		HWTILE			: "hwTile",
		ISOBJECT		: "isObject",
		EXITUPLVL		: "exitUpLevel",
		EXITDOWNLVL		: "exitDownLevel",
		EXITLEFTLVL		: "exitLeftLevel",
		EXITRIGHTLVL	: "exitRightLevel"
	},
	gameObject: {				// Name of each attribute on the Tiled 'Header Object layer'
								// Note: Sets can be reordered, but objects within a set must be kept in this sequence
		mummyWhite			: objIdx++,
		mummyOrange			: objIdx++,
		mummyBlue			: objIdx++,
		mummyRed			: objIdx++,
		mummyYellow			: objIdx++,

		exitAuto			: objIdx++,
		exitUp				: objIdx++,
		exitDown			: objIdx++,
		exitLeft			: objIdx++,
		exitRight			: objIdx++,

		flipDoorRL			: objIdx++,
		flipDoorLR			: objIdx++,
		flipDoorInverter	: objIdx++,

		trapWall			: objIdx++,

		// All objects made of single tiles have hardcoded values <=0x80
		blank				: 0x00,
		hole				: 0x0E,
		leapOfFaith			: 0x10,
		doubleBrick			: 0x13,
		hardBrick			: 0x19,
		singleHardBrick		: 0x1A,
		sword				: 0x30,
		orbBlue				: 0x43,
		orbCyan				: 0x44,
		orbPurple			: 0x45,
		orbYellow			: 0x46,
		orbGreen			: 0x47,
		orbGray				: 0x48,
		pickAxe				: 0x80
	},

	// === Auxiliary functions ===
	asciiEncoder: function( myString ){
		// Since TextEncoder doesn't work, I had to reinvent the wheel
		var myArray = [];
		for (i=0; i < myString.length ; i++) {
			myArray.push( myString.charCodeAt(i) );
		}
		return myArray;
	},
	clamp: function (num, min, max){
			return Math.min(Math.max(num, min), max);
	},
	xyToBytes: function( xt, yt ){
		let coordArray = [];
		let x = xt*8;
		let room = this.clamp( x>>8, 0, 2 );		// Be sure that no invalid rooms are used as it causes glitches

		coordArray.push( (yt+1)*8&0xFF );
		coordArray.push( x&0xFF, room );
		return coordArray;
	},
	debugLog: function() {
		if( this.debugVerbose )
			console.log( ...arguments );
	},

	// === Auxiliary global variables ===
	alertProperties	: true,					// Avoid pestering the user. Stop alerting if requested.
	myFileName		= __filename				// Saves the script filename/path, in case it's needed later
}

// ==== Main body ====
var customMapFormat1 = {
    name: kv1gam.description,
    extension: kv1gam.fileExtension,

	// Write KVG
    write: function( map, fileName ){
		 function cellTileId( xt , yt ){
			let tileId = layer.cellAt(xt, yt).tileId;
			let tileType = tileSet.tile( tileId ).type;
			return kv1gam.gameObject[ tileType ];
		}
		console.info(`\n\n${"-".repeat(64)}\nExporting to ${fileName}` );
		var rtc = new Date();
        var hdr				= [];	// Used to collect the header data
        var mmd				= [];	// Used to collect the main map data
		var tail			= [];	// Used to collect the tail data
		var exitDoors 		= [];
		var exitDoorLevel	= Array(4).fill(0)
		var enemiesBytes 	= Array(1).fill(0);
		var orbsBytes 		= Array(1).fill(0);
		var swordsBytes 	= Array(1).fill(0);
		var pickAxesBytes 	= Array(1).fill(0);
		var flipDoorsBytes 	= Array(1).fill(0);
		var trapWallsBytes 	= Array(1).fill(0);
		var tileSet 		= map.usedTilesets()[0];

		// === Part 1: Gather and validate all data ===

		// --- Map tile data ---
		for (let i = 0; i < map.layerCount; ++i) {
			var layer = map.layerAt(i);

			if ( layer.isTileLayer && layer.name == kv1gam.layerNames.MAINMAP ) {
				kv1gam.debugLog( `Found: ${layer.name}. Layer: ${i}` );		
				if( !( kv1gam.mapHeights.includes( layer.height ) && kv1gam.mapWidths.includes( layer.width )) ){
					tiled.error(`${kv1gam.displayName} expected widths: ${kv1gam.mapWidths}, expected heights: ${kv1gam.mapHeights}. Found: Xt=${layer.width},Yt=${layer.height} `);
					return ( `${kv1gam.displayName} incorrect layer size!`);
				}

				for( let yt = 0; yt < layer.height; ++yt ){
					for( let xt = 0; xt < layer.width; ++xt ){
						if( xt > 0 && xt < ( layer.width-1 )){
							var tileId = layer.cellAt(xt, yt).tileId;
							var hwTile = kv1gam.clamp( tileSet.tile( tileId ).property( kv1gam.attrNames.HWTILE ),0, kv1gam.maxTile );
							var isObject = tileSet.tile( tileId ).property( kv1gam.attrNames.ISOBJECT );
						} else { 												// Force hard bricks on the 1st and last columns
							var hwTile = kv1gam.gameObject.hardBrick;
							var isObject = false;
						}

						if( isObject ){
							let objType = tileSet.tile( tileId ).type;
							let objId = kv1gam.gameObject[ objType ];
							mmd.push( kv1gam.gameObject.blank );				// Put an empty tile on this cell
							switch( objId ){
								// Exit Doors
								case kv1gam.gameObject.exitAuto:
								case kv1gam.gameObject.exitUp:
								case kv1gam.gameObject.exitDown:
								case kv1gam.gameObject.exitLeft:
								case kv1gam.gameObject.exitRight:
									if( exitDoors.length < 4 ){
										// Skip the connectionDoor tiles
										if( !tileSet.tile( layer.cellAt(xt-1, yt).tileId ).property( kv1gam.attrNames.ISOBJECT ) ){
											let connectDoor = tileSet.tile( layer.cellAt(xt+1, yt).tileId ).type;
											kv1gam.debugLog( `map: object type=${objType}   \t: Xt=${xt},\tYt=${yt}, connectDoor=${connectDoor}` );
											exitDoors.push({ xt: xt, yt: yt, id: objType, connectDoor: connectDoor });
										}
									} else
										tiled.warn( `${kv1gam.displayName}: Maximum of ${exitDoors.length} exit doors reached. Exit door at (${xt},${yt}) will be discarded.` );
									break;	
								// Enemies
								case kv1gam.gameObject.mummyWhite:
								case kv1gam.gameObject.mummyOrange:
								case kv1gam.gameObject.mummyYellow:
								case kv1gam.gameObject.mummyBlue:
								case kv1gam.gameObject.mummyRed:
									if( enemiesBytes[0] < 5 ){
										kv1gam.debugLog( `map: object type=${objType}   \t: Xt=${xt},\tYt=${yt}` );
										enemiesBytes.push( ...kv1gam.xyToBytes( xt, yt ), kv1gam.gameObject[ objType ]-kv1gam.gameObject.mummyWhite );
										enemiesBytes[0]++;
									} else
									tiled.warn( `${kv1gam.displayName}: Maximum of ${enemiesBytes[0]} enemies reached. Enemy at (${xt},${yt}) will be discarded.` );
									break;
								// Orbs
								case kv1gam.gameObject.orbBlue:
								case kv1gam.gameObject.orbCyan:
								case kv1gam.gameObject.orbPurple:
								case kv1gam.gameObject.orbYellow:
								case kv1gam.gameObject.orbGreen:
								case kv1gam.gameObject.orbGray:
									if( orbsBytes[0] < 16 ){
										kv1gam.debugLog( `map: object type=${objType}   \t: Xt=${xt},\tYt=${yt}` );
										let orbNum = (kv1gam.gameObject[ objType ]-kv1gam.gameObject.orbBlue)&0x0F;
										orbsBytes.push( orbNum*16+0x31, ...kv1gam.xyToBytes( xt, yt ));
										orbsBytes[0]++;
									} else
										tiled.warn( `${kv1gam.displayName}: Maximum of ${orbsBytes[0]} orbs reached. Orb at (${xt},${yt}) will be discarded.` );
									break;
								// Tools
								case kv1gam.gameObject.sword:
									if( swordsBytes[0] < 6 ){
										kv1gam.debugLog( `map: object type=${objType}   \t: Xt=${xt},\tYt=${yt}` );
										swordsBytes.push( ...kv1gam.xyToBytes( xt, yt ));
										swordsBytes[0]++;
									} else
										tiled.warn( `${kv1gam.displayName}: Maximum of ${swordsBytes[0]} swords reached. Sword at (${xt},${yt}) will be discarded.` );
									break;
								case kv1gam.gameObject.pickAxe:
									if( pickAxesBytes[0] < 16 ){
										kv1gam.debugLog( `map: object type=${objType}   \t: Xt=${xt},\tYt=${yt}` );
										pickAxesBytes.push( ...kv1gam.xyToBytes( xt, yt ));
										pickAxesBytes[0]++;
									} else
										tiled.warn( `${kv1gam.displayName}: Maximum of ${pickAxesBytes[0]} pickaxes reached. Pickaxe at (${xt},${yt}) will be discarded.` );

									break;
								// Flip Doors
								case kv1gam.gameObject.flipDoorLR:
									xOffsetInv = 1;
									xOffsetDoor = 0;
								case kv1gam.gameObject.flipDoorRL:
									if( objId == kv1gam.gameObject.flipDoorRL ){
										xOffsetInv = -1;
										xOffsetDoor = -1;
									}
									if( cellTileId( xt, yt-1 ) != objId ){
										if( flipDoorsBytes[0] < 31 ){
											kv1gam.debugLog( `map: object type=${objType}   \t: Xt=${xt},\tYt=${yt}` );
											let autoInvert = (cellTileId( xt+xOffsetInv, yt ) == kv1gam.gameObject.flipDoorInverter?1:0);
											for( height = 2 ; ( cellTileId( xt, yt+height )) == objId && (height++ < 5 ); );
											flipDoorsBytes.push( ((height-2)<<1)+autoInvert, ...kv1gam.xyToBytes( xt+xOffsetDoor, yt ), objId == kv1gam.gameObject.flipDoorRL?4:8 );
											flipDoorsBytes[0]++;
										} else
											tiled.warn( `${kv1gam.displayName}: Maximum of ${flipDoorsBytes[0]} flip-doors reached. Flip-door at (${xt},${yt}) will be discarded.` );
									}
									break;
								// Trap Walls
								case kv1gam.gameObject.trapWall:
									if( cellTileId( xt, yt+1 ) == objId && cellTileId( xt, yt+2 ) != objId ){
										if( trapWallsBytes[0] < 17 ){
											kv1gam.debugLog( `map: object type=${objType}   \t: Xt=${xt},\tYt=${yt}` );
											trapWallsBytes.push( ...kv1gam.xyToBytes( xt, yt ));
											trapWallsBytes[0]++;
										} else
											tiled.warn( `${kv1gam.displayName}: Maximum of ${trapWallsBytes[0]} trap-walls reached. Trap-wall at (${xt},${yt}) will be discarded.` );
									}	
									break;
							}
						} else mmd.push( hwTile );
					}  
					if( layer.width < Math.max( ...kv1gam.mapWidths ))
						mmd = mmd.concat( Array( Math.max( ...kv1gam.mapWidths ) - layer.width ).fill( 0 ));
				}
				mmd=mmd.concat( Array( Math.max( ...kv1gam.mapWidths )).fill( 0xFF ));		// Add an empty line at the end
			}
		}
		kv1gam.debugLog( "\n\nenemiesBytes:", enemiesBytes );
		kv1gam.debugLog( "orbsBytes:", orbsBytes );
		kv1gam.debugLog( "swordsBytes:", swordsBytes );
		kv1gam.debugLog( "pickAxesBytes:", pickAxesBytes );
		kv1gam.debugLog( "flipDoorsBytes:", flipDoorsBytes );
		kv1gam.debugLog( `trapWallsBytes: [${trapWallsBytes}]\n\n` );

		if( exitDoors.length == 0 ){
			let message=`${kv1gam.displayName}: The level must have at least one exit door!`;
			tiled.error( message );
			return (  message );
		}
		var exitDoorBytes = Array( Math.max( 4*5 )).fill( 0xFF );

		exitDoorLevel[0] = kv1gam.clamp( map.properties()[ kv1gam.attrNames.EXITUPLVL ], 1, 63 );
		exitDoorLevel[1] = kv1gam.clamp( map.properties()[ kv1gam.attrNames.EXITDOWNLVL ], 1, 63 );
		exitDoorLevel[2] = kv1gam.clamp( map.properties()[ kv1gam.attrNames.EXITLEFTLVL ], 1, 63 );
		exitDoorLevel[3] = kv1gam.clamp( map.properties()[ kv1gam.attrNames.EXITRIGHTLVL ], 1, 63 );

		// Heuristic to fit the doors and output the array in binary format
		exitDoors.forEach( function( door ) {	// 1st pass: Fit the assigned doors
			let doorNum = (kv1gam.gameObject[ door.id ]-kv1gam.gameObject.exitUp);
			if( doorNum >=0 ){
				let gotoLevel = exitDoorLevel[doorNum];	
				let connectDoor = kv1gam.gameObject[ door.connectDoor ]==kv1gam.gameObject.exitAuto? doorNum^1 : (kv1gam.gameObject[ door.connectDoor ]-kv1gam.gameObject.exitUp);
				kv1gam.debugLog( `exitDoor pass1: type=${door.id},\tindex=${doorNum}, Xt=${door.xt},\tYt=${door.yt},\tgotoLevel=${gotoLevel}, connectDoor=${connectDoor}` );
				exitDoorBytes[doorNum*5+0] = (door.yt+1)*8;
				exitDoorBytes[doorNum*5+1] = door.xt*8&0xFF;
				exitDoorBytes[doorNum*5+2] = door.xt*8>>8;
				exitDoorBytes[doorNum*5+3] = exitDoorLevel[doorNum];
				exitDoorBytes[doorNum*5+4] = 2**connectDoor;
			}
		});
		exitDoors.forEach( function( door ) {	// 2nd pass: Fit the auto doors on the remaining spaces following the raster order
			let assigned = false;
			let doorNum = (kv1gam.gameObject[ door.id ]-kv1gam.gameObject.exitUp);
			for ( let i=0; !assigned && doorNum <0 && i<4*5; i=i+5 ){
				if( exitDoorBytes[ i ] == 0xFF ){
					let gotoLevel = exitDoorLevel[i/5|0];
					let connectDoor = kv1gam.gameObject[ door.connectDoor ]==kv1gam.gameObject.exitAuto? doorNum^1 : (kv1gam.gameObject[ door.connectDoor ]-kv1gam.gameObject.exitUp);
					kv1gam.debugLog( `exitDoor pass2: type=${door.id},\tindex=${i/5|0}, Xt=${door.xt},\tYt=${door.yt},\tgotoLevel=${gotoLevel}, connectDoor=${connectDoor}` );
					exitDoorBytes[i+0] = (door.yt+1)*8;
					exitDoorBytes[i+1] = door.xt*8&0xFF;
					exitDoorBytes[i+2] = door.xt*8>>8;
					exitDoorBytes[i+3] = gotoLevel;
					exitDoorBytes[i+4] = 2**connectDoor;
					assigned = true;
				}
			}
		});

		kv1gam.debugLog( `exitDoorBytes: ${exitDoorBytes}\n\n` );

		// --- File header ---
		let author			= map.properties()[ kv1gam.attrNames.AUTHOR ].substring(0, 3);
		let title			= map.properties()[ kv1gam.attrNames.TITLE ].substring(0, 19);
		let theme			= kv1gam.clamp( map.properties()[ kv1gam.attrNames.THEME ], 0, 3);

		kv1gam.debugLog( "Author:", author );
		kv1gam.debugLog( "Title :", title );
		kv1gam.debugLog( "Theme :", theme );

		// Build the header with the data we collected
		hdr = kv1gam.asciiEncoder ( kv1gam.magicID );		//  Unique magicID
		hdr.push( theme );
		hdr.push( 0 );		// Reserved for future use: stageSong
		hdr = hdr.concat( kv1gam.asciiEncoder( author.padEnd(3,'\0') ) );
		hdr = hdr.concat( kv1gam.asciiEncoder( title.padEnd(19,'\0') ) );		
		hdr = hdr.concat( Array( 64 ).fill( 0 ));		// Pad to fill 1 line

		// --- File tail ---
		tail.push( ...exitDoorBytes, ...enemiesBytes, ...orbsBytes, ...swordsBytes, ...pickAxesBytes, ...flipDoorsBytes, ...trapWallsBytes );
		tail.push( 0x00, 0x00, 0x01, 0xFF );		// EOF signature
		if( tail.length > kv1gam.tailLength ){
			tiled.error( `${kv1gam.displayName}: Object memory overflow. Please reduce the number of objects.` );
			return( `${kv1gam.displayName}: Object memory overflow.\nPlease Reduce the number of objects.` );
		} else if( tail.length < kv1gam.tailLength ) 
			tail = tail.concat( Array( kv1gam.tailLength-tail.length ).fill(0) );	// tail padding

		if( hdr.length != kv1gam.headerLength ){
			tiled.error(kv1gam.displayName+" expected header lenght:"+kv1gam.headerLength+", found:"+hdr.length );
			return( `${kv1gam.displayName}:  Header layer error!` );
		}
		if( mmd.length != kv1gam.mapLength ){
			tiled.error(kv1gam.displayName+" expected map lenght:"+kv1gam.mapLength+", found:"+mmd.length );
			return( `${kv1gam.displayName}: Tile Map layer error!` );
		}

		// === Part 2: Save the file ===
		var file = new BinaryFile(fileName, BinaryFile.WriteOnly);
		var savhdr = new Uint8Array(hdr).buffer;
		var savmap = new Uint8Array(mmd).buffer;
		var savtail = new Uint8Array(tail).buffer;

		file.write( savhdr );
		file.write( savmap );
		file.write( savtail );
		file.commit();
		console.info(`Export ${kv1gam.fileExtension} format completed at ${rtc.toLocaleTimeString()}.`);
    },


	// ----------------------------------------------------------------------------------------------
	// Read KVG
	read: function( fileName ) {
		console.info(`\n\n${"-".repeat(64)}\nImporting ${fileName}` );
		var file = new BinaryFile(fileName, BinaryFile.ReadOnly);
		var fileData = file.readAll();
		var fileBytes = new Uint8Array( fileData );
		file.close();

		// Check the header signature, if necessary
		if( kv1gam.magicID.length > 0 ) {
			var magicID	= String.fromCharCode( ...fileBytes.slice(  0, kv1gam.magicID.length )).replace(/\x00/g, '');
			if( magicID != kv1gam.magicID ){
				let forceLoad = tiled.confirm( `Warning: This doesn't appear to be a valid ${kv1gam.fileExtension} file.\n\n Do you wish to force to load it?`, kv1gam.description );
				if( !forceLoad ){
					tiled.error("Invalid "+kv1gam.fileExtension+" file.");
					//tiled.trigger("ViewIssues");		// Wishlist
				return;
				}
			}
		}

		// Find the number of used rooms
		var rightMostUsedTile = 0;
		for( let yt = 0; yt < kv1gam.mapHeights[0]; ++yt) {
			for( let xt = 0; xt < kv1gam.mapWidths[2]; ++xt) {
				let tileID = fileBytes[kv1gam.headerLength+96*yt+xt];
				if( tileID > 0 && xt > rightMostUsedTile )
					rightMostUsedTile = xt;
			}
		}
		kv1gam.debugLog( `Rightmost used tile=${rightMostUsedTile}, number of rooms=${Math.ceil( rightMostUsedTile/32 )}` );
		var mapWidth = kv1gam.mapWidths[ Math.ceil( rightMostUsedTile/32 )-1 ];

		var exitDoors = fileBytes.slice(0x900, 0x914);
		var levelObjects = fileBytes.slice(0x914, -1 );

		// Get the header attributes
		var theme	= fileBytes[8];
		// var song	= fileBytes[9];		// Reserved for future use
		var author	= String.fromCharCode( ...fileBytes.slice(  10, 13 )).replace(/\x00/g, '');
		var title	= String.fromCharCode( ...fileBytes.slice( 13, 32 )).replace(/\x00/g, '');
		var exitUpLevel		= fileBytes[0x903]<255? fileBytes[0x903]:1;
		var exitDownLevel	= fileBytes[0x908]<255? fileBytes[0x908]:1;
		var exitLeftLevel	= fileBytes[0x90D]<255? fileBytes[0x90D]:1;
		var exitRightLevel	= fileBytes[0x912]<255? fileBytes[0x912]:1;

		// Create the map
		var map = new TileMap();
        map.setSize( mapWidth, kv1gam.mapHeights[0]);
        map.setTileSize( 8, 8);
        map.orientation = TileMap.Orthogonal;
		map.backgroundColor = "#303030";
		map.layerDataFormat = "CSV";
		map.setProperty( kv1gam.attrNames.THEME,		theme);
		// map.setProperty( kv1gam.attrNames.SONG,		song);
		map.setProperty( kv1gam.attrNames.AUTHOR,		author);
		map.setProperty( kv1gam.attrNames.TITLE,		title);
		map.setProperty( kv1gam.attrNames.EXITUPLVL, 	exitUpLevel);
		map.setProperty( kv1gam.attrNames.EXITDOWNLVL,	exitDownLevel);
		map.setProperty( kv1gam.attrNames.EXITLEFTLVL,	exitLeftLevel);
		map.setProperty( kv1gam.attrNames.EXITRIGHTLVL,	exitRightLevel);

		// Attach a Tile set
		var tileSet =  tiled.open( "ext:"+kv1gam.tileSetName );
		var widthTiles = tileSet.imageWidth/tileSet.tileWidth|0;
		var hwTileSet = [];
		var gameObjIdxSet = [];
        if( tileSet && tileSet.isTileset ){
			map.addTileset( tileSet );
			for( let i = 0; i < tileSet.tileCount ; ++i){			// Index the tileSet by the hardware indexes
				let isObject = tileSet.tile( i ).property( kv1gam.attrNames.ISOBJECT );
				let hwTile = tileSet.tile( i ).property( kv1gam.attrNames.HWTILE );
				if( isObject ){
					let objId = tileSet.tile( i ).type;
					kv1gam.debugLog( "tileSet object:", i, objId );
					gameObjIdxSet[ kv1gam.gameObject[ objId ] ] = i;
					if( hwTile >=0 )
						hwTileSet[ hwTile ] = i;
				}
				else {
					hwTileSet[ hwTile ] = i;
				}
			}
		} else
			tiled.error("Invalid Tile Set:", kv1gam.tileSetName );

		
		// Create the Tile layer
		var tileLayer 		= new TileLayer();
		tileLayer.width 	= map.width;
		tileLayer.height	= map.height;
		tileLayer.name		= kv1gam.layerNames.MAINMAP;
		var layerEdit		= tileLayer.edit();

		// Copy background data to the layer
		for( let y = 0; y < map.height; ++y) {
			for( let x = 0; x < map.width; ++x) {
				let tileID = fileBytes[kv1gam.headerLength+96*y+x];
				if( tileID <= kv1gam.maxTile ) 
					layerEdit.setTile( x, y, tileSet.tile( hwTileSet[ tileID ]));
				else
					layerEdit.setTile( x, y, tileSet.tile( hwTileSet[ kv1gam.gameObject.blank ]));
			}
		}
		// Crop all specific lists from the objects list
		let i = 0;
		var enemies		= levelObjects.slice( i+1, i+levelObjects[i++]*4+1 );
		i += enemies.length;
		kv1gam.debugLog ("orb index, end:", i, i+levelObjects[i]*4+1 );
		var orbs		= levelObjects.slice( i+1, i+levelObjects[i++]*4+1 );
		kv1gam.debugLog ("orbs:", orbs);
		i += orbs.length;

		var numSwords = levelObjects[i++];		// Number of swords
		var tools		= [];
		tools.push( ...levelObjects.slice( i, i+numSwords*3 ));
		kv1gam.debugLog ("tools: swords=", tools );
		i += tools.length;
		var numPickAxes = levelObjects[i++];	// Number of pickAxes
		tools.push( ...levelObjects.slice( i, i+numPickAxes*3 ));
		kv1gam.debugLog ("tools: swords+pickAxes=", tools );
		i +=numPickAxes*3;

		var flipDoors	= levelObjects.slice( i+1, i+levelObjects[i++]*5+1 );
		i += flipDoors.length;
		var trapWalls	= levelObjects.slice( i+1, i+levelObjects[i++]*3+1 );
		i += trapWalls.length;

		// Render the enemies to the layer
		for( let i = 0 ; i < enemies.length ; i=i+4 ){
			let enemyY = (enemies[ i ]>>3)-1;
			let enemyX = (enemies[ i+1 ]+256*enemies[ i+2 ])>>3;
			let enemyType = kv1gam.clamp( enemies[ i+3 ], 0, 4 );
			kv1gam.debugLog( `enemy Type=${enemyType}, X=${enemyX}, Y=${enemyY}` );
			for( let y = 0 ; y < 2 ; ++y ){
				for( let x = 0 ; x < 2 ; ++x ){
					layerEdit.setTile( enemyX+x, enemyY+y, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.mummyWhite+enemyType ]+x+y*widthTiles ));
				}
			}
		}

		// Render the orbs to the layer
		for( let i = 0 ; i < orbs.length ; i=i+4 ){
			let orbY = (orbs[ i+1 ]>>3)-1;
			let orbX = (orbs[ i+2 ]+256*orbs[ i+3 ])>>3;
			let orbType = kv1gam.gameObject.orbBlue+kv1gam.clamp( (orbs[ i ]>>4)-3, 0, 5 );
			kv1gam.debugLog( "orb Type,X,Y:", orbType, orbX, orbY );
			layerEdit.setTile( orbX, orbY, tileSet.tile( gameObjIdxSet[ orbType ]));
		}

		// Render the tools to the layer
		for( let i = 0 ; i < tools.length ; i=i+3 ){
			let toolY	= (tools[ i+0 ]>>3)-1;
			let toolX	= (tools[ i+1 ]+256*tools[ i+2 ])>>3;
			let tileID	= (i/3)<numSwords? gameObjIdxSet[ kv1gam.gameObject.sword ]:gameObjIdxSet[ kv1gam.gameObject.pickAxe ];
			kv1gam.debugLog( "tool Type,X,Y:", tileID, toolX, toolY );
			layerEdit.setTile( toolX, toolY, tileSet.tile( tileID ));
		}

		// Render the flipDoors to the layer
		for( let i = 0 ; i < flipDoors.length ; i=i+5 ){
			let height 		= kv1gam.clamp( (flipDoors[ i+0 ]>>1)+2, 2, 4 );
			let type 		= kv1gam.clamp( (flipDoors[ i+4 ]>>2)-1, 0, 1 );
			let autoInvert	= (flipDoors[ i+0 ]&1)>0;
			let yt 			= (flipDoors[ i+1 ]>>3)-1;
			let xt 			= ((flipDoors[ i+2 ]+256*flipDoors[ i+3 ])>>3)+(1-type);
			kv1gam.debugLog( "flipDoor type,X,Y,height:", type, xt, yt, height );
			for( let y = 0 ; y < height ; ++y ){
				for( let x = 0 ; x < 2 ; ++x ){
					let xOffset = (type>0?x:-x);
					let tile = ( autoInvert && y==0 && x==1 )?
						tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.flipDoorInverter ]):
						tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.flipDoorRL+type ]+xOffset );
					layerEdit.setTile( xt+xOffset, yt+y, tile );
				}
			}
		}

		// Render the trapWalls to the layer
		for( let i = 0 ; i < trapWalls.length ; i=i+3 ){
			let trapWallY = (trapWalls[ i+0 ]>>3)-1;
			let trapWallX = (trapWalls[ i+1 ]+256*trapWalls[ i+2 ])>>3;
			kv1gam.debugLog( "trapWall X,Y:", trapWallX, trapWallY );
			for( let y = trapWallY+1 ; 
				y >= 0 &&
				fileBytes[kv1gam.headerLength+96*y+trapWallX] == 0;
				--y ){
					kv1gam.debugLog ("y=", y, fileBytes[kv1gam.headerLength+96*y+trapWallX] );
					layerEdit.setTile( trapWallX, y, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.trapWall ] ));
			}
		}		

		// Render the four Exit Doors to the layer
		for( let i = 0; i < 4 ; ++i ){
			let doorY = exitDoors[ 5*i+0 ];
			let doorYt = (doorY>>3)-1;
			let doorXt = (exitDoors[ 5*i+1 ]+256*exitDoors[ 5*i+2 ])>>3;
			let doorDestLevel = exitDoors[ 5*i+3 ];
			let doorConnection = Math.log2( exitDoors[ 5*i+4 ] )|0;
			if( doorY < 0xFF ){  // Is this door enabled?
				kv1gam.debugLog( `exitDoor Type=${i}, xt=${doorXt}, yt=${doorYt}, destLevel=${doorDestLevel}, connection=${doorConnection}` );
				for( let y = -1 ; y < 2 ; ++y ){
					for( let x = -2 ; x < 3 ; ++x ){
						layerEdit.setTile( doorXt+x, doorYt+y, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.exitAuto ]+x+y*widthTiles ));
					}
				}
			}
			// Set the door direction icons
			layerEdit.setTile( doorXt, doorYt, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.exitUp+i ] ));
			layerEdit.setTile( doorXt+1, doorYt, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.exitUp+doorConnection ] ));
			// Math.log2( fileBytes[0x904])|0

	}

		layerEdit.apply();
		map.addLayer( tileLayer );
		tiled.close( tileSet );			

		if ( kv1gam.alertProperties )
			kv1gam.alertProperties = tiled.confirm( "File loaded.\r\n\r\nRemember to set the Author, Theme, Title and exit destination levels on Map→Map Properties\r\n\r\nRemind this again on this session?", kv1gam.description );

		return map;
	}
}

//=====================================================================================

/*	KV1-Plus Editor file "EDATAx" format (this is different from the game .DAT level format)
	offset
	+000h,+001h	: 0Bh, 0Fh signature
	+002h,+003h	: ?
	+004h,+005h	: y,x of the Exit Door
	+006h			: Index of the 1st enemy on the enemy array at +07h: Must be 1, otherwise the game shows bugs on enemy spawn order
	+007h,+008h	: y,x of the 3rd Mummy
	+009h,+00Ah	: y,x of the 1st Mummy
	+00Bh,+00Ch	: y,x of the 2nd Mummy
	+00Dh~+00Fh	: 3Ah, 80h, DCh signature?
	+010h,+011h	: RAM variable, used to detect CTRL+Q (useless)
	+012h,+013h	: ?
	+020h,+021h	: ?
	+022h~+02Fh	: Unused space?
	+020h~+03Fh	: Out-of-screen map line. Must be empty.
	+050h~+30Fh	: Tile map 32x22
	+310h~+32Fh	: Filled with 00h
*/

var kv1ed = {
	// === File-format Configuration  ===
	displayName		: "KV1PlusEditor",			// Name of this file format to be displayed on Tiled
	description		: "King's Valley 1 Plus Editor",	// Description of this file format to be displayed on Tiled
	fileExtension	: "KVE",					// File extension
	magicID			: "\x0B\x0F",				// Unique Magic ID on the file header
	headerLength	: 0x50,						// Header length
	mapLength		: 32*22+32,					// Map lenght
	maxTile			: 128,						// Highest allowed tile number
	tileSetName		: "KV1TileSet.tsx",			// Tileset to use with this format. Must be present on the Tiled Extensions folder.
	layerNames: { 								// Name of each layer used on Tiled
		MAINMAP: "Tile Map"
	},
	// === Auxiliary global variables ===
	alertProperties	: true,					// Avoid pestering the user. Stop alerting if requested.
	myFileName = __filename					// Saves the script filename/path, in case it's needed later
}

// ==== Main body ====
var customMapFormat2 = {
    name: kv1ed.description,
    extension: kv1ed.fileExtension,

	// Read KVE
	read: function( fileName ) {
		console.info(`\n\n${"-".repeat(64)}\nImporting ${fileName}` );
		var file = new BinaryFile(fileName, BinaryFile.ReadOnly);
		var fileData = file.readAll();
		var fileBytes = new Uint8Array( fileData );
		file.close();

		// Check the header signature, if necessary
		if( kv1ed.magicID.length > 0 ) {
			var magicID	= String.fromCharCode( ...fileBytes.slice(  0, kv1ed.magicID.length )).replace(/\x00/g, '');
			if( magicID != kv1ed.magicID ){
				tiled.error("Invalid "+kv1ed.fileExtension+" file.");
				//tiled.trigger("ViewIssues");		// Wishlist
				return;
			}
		}

		// Get the header attributes
		var theme	= fileBytes[7]<4? fileBytes[7]:0;
		// var song	= fileBytes[8];
		var author	= "";	// String.fromCharCode( ...fileBytes.slice(  9, 12 )).replace(/\x00/g, '');
		var title	= "";	// String.fromCharCode( ...fileBytes.slice( 12, 32 )).replace(/\xFF/g, '');

		// Create the map
		var map = new TileMap();
        map.setSize( 32, 22);
        map.setTileSize( 8, 8);
        map.orientation = TileMap.Orthogonal;
		map.backgroundColor = "#303030";
		map.layerDataFormat = "CSV";
		map.setProperty( kv1gam.attrNames.THEME, theme);
		// map.setProperty( kv1gam.attrNames.SONG, song);
		map.setProperty( kv1gam.attrNames.AUTHOR, author);
		map.setProperty( kv1gam.attrNames.TITLE, title);
		map.setProperty( kv1gam.attrNames.EXITUPLVL, 	1);
		map.setProperty( kv1gam.attrNames.EXITDOWNLVL,	1);
		map.setProperty( kv1gam.attrNames.EXITLEFTLVL,	1);
		map.setProperty( kv1gam.attrNames.EXITRIGHTLVL,	1);

		// Attach a Tile set
		var tileSet =  tiled.open( "ext:"+kv1ed.tileSetName );
		var widthTiles = tileSet.imageWidth/tileSet.tileWidth|0;
		var hwTileSet = [];	
		var gameObjIdxSet = [];

        if( tileSet && tileSet.isTileset ){
			map.addTileset( tileSet );
			for( let i = 0; i < tileSet.tileCount ; ++i){			// Index the tileSet by the hardware indexes
				let isObject = tileSet.tile( i ).property( kv1gam.attrNames.ISOBJECT );
				let hwTile = tileSet.tile( i ).property( kv1gam.attrNames.HWTILE );
				if( isObject ){
					let objId = tileSet.tile( i ).type;
					kv1gam.debugLog( "tileSet object:", i, objId );
					gameObjIdxSet[ kv1gam.gameObject[ objId ] ] = i;
					if( hwTile >=0 )
						hwTileSet[ hwTile ] = i;
				}
				else {
					hwTileSet[ hwTile ] = i;
				}
			}
		} else
			tiled.error("Invalid Tile Set:", kv1ed.tileSetName );

		
		// Create the Tile layer
		var tileLayer 		= new TileLayer();
		tileLayer.width 	= map.width;
		tileLayer.height	= map.height;
		tileLayer.name		= kv1ed.layerNames.MAINMAP;
		var layerEdit		= tileLayer.edit();

		// Copy background data to the layer
		for( let y = 0; y < map.height; ++y) {
			for( let x = 0; x < map.width; ++x) {
				let tileID = fileBytes[kv1ed.headerLength+32*y+x];
				if( tileID <= kv1ed.maxTile ) 
					layerEdit.setTile( x, y, tileSet.tile( hwTileSet[ tileID ] ));
				else
					layerEdit.setTile( x, y, tileSet.tile( hwTileSet[ 0 ] ));
			}
		}
		// Render the enemy data to the layer
		for( let i = 0 ; i < 3 ; ++i ){
			let enemyY = (fileBytes[ 7+i*2 ]>>3)-1;
			let enemyX = fileBytes[ 8+i*2 ]>>3;
			let enemyType = fileBytes[ 13+i ]<5?fileBytes[ 13+i ]:0;
			kv1gam.debugLog( `enemy Type=${enemyType}, X=${enemyX}, Y=${enemyY}` );
			for( let y = 0 ; y < 2 ; ++y ){
				for( let x = 0 ; x < 2 ; ++x ){
					if( enemyY < 26 )	// Is this enemy enabled?
						layerEdit.setTile( enemyX+x, enemyY+y, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.mummyWhite+enemyType ]+x+y*widthTiles ));
				}
			}
		}
	
		// Render the Exit Door location to the layer
		let doorYt = (fileBytes[ 4 ]>>3)-1;
		let doorXt = fileBytes[ 5 ]>>3;
		for( let y = -1 ; y < 2 ; ++y ){
			for( let x = -2 ; x < 3 ; ++x ){
				layerEdit.setTile( doorXt+x, doorYt+y, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.exitAuto ]+x+y*widthTiles ));
			}
		}
		// Set the door direction icons. It's hardcoded in this format
		layerEdit.setTile( doorXt, doorYt, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.exitRight ] ));
		layerEdit.setTile( doorXt+1, doorYt, tileSet.tile( gameObjIdxSet[ kv1gam.gameObject.exitLeft ] ));

		layerEdit.apply();
		map.addLayer( tileLayer );
		tiled.close( tileSet );			

		if ( kv1gam.alertProperties )
			kv1gam.alertProperties = tiled.confirm( "File loaded.\r\n\r\nRemember to set the Author, Theme, Title and exit destination levels on Map→Map Properties\r\n\r\nRemind this again on this session?", kv1gam.description );

		return map;
	}
}

// ==== Register this plugin ====
tiled.registerMapFormat(kv1gam.displayName, customMapFormat1)
tiled.registerMapFormat(kv1ed.displayName, customMapFormat2)

