/** Yazzie MSX Remastered Custom Level Export plugin for Tiled 
 * By FRS, 2021. Based on:
 * - videogame-format.js by diogoeichert
 * - "How to write a custom Text File importer for Tiled Map Editor" guide, by Mezzmer & eishiya
 * 
 * License: CC:BY-SA
 * 
 * This file was designed as an easy-to-adapt template for other 8bit projects
 * Special thanks for MsxKun@msx.org, for giving me the pointers on where to begin
 * 
 * Install this script as explained in the following article:
 * https://doc.mapeditor.org/en/stable/reference/scripting/#binaryfile
*/

// ==== Auxiliary structures and functions for this file format ====
var yaz = {
	// === File-format Configuration  ===
	displayName		: "Yazzie",					// Name of this file format to be displayed on Tiled
	description		: "Yazzie Custom Level",	// Description of this file format to be displayed on Tiled
	fileExtension	: "yaz",					// File extension
	magicID			: "Yazzie1",				// Unique Magic ID on the file header
	headerLength	: 32,						// Header length
	maxTile			: 34,						// Highest allowed tile number, since Tiled only suports multiples of the TileSet width
	tileSetName		: "YazzieTileSet1.tsx",		// Tileset to use with this format. Must be located on the same dir.
	layerNames: { 								// Name of each layer used on Tiled
		MAINMAP: "Tile Map"
	},
	attrNames: {				// Name of each attribute on the Tiled 'Header Object layer'
		THEME:	"Theme",
		SONG:	"Song",
		AUTHOR:	"Author",
		TITLE:	"Title"
	},

	// === Auxiliary functions ===
	asciiEncoder: function( myString ){
		// Since TextEncoder doesn't work, I had to reinvent the wheel
		var myArray = [];
		for (i=0; i < myString.length ; i++) {
			myArray.push( myString.charCodeAt(i) );
		}
		return myArray;
	},
	clamp: function (num, min, max){
			return Math.min(Math.max(num, min), max);
	},

	// === Auxiliary global variables ===
	alertProperties	: true					// Avoid pestering the user. Stop alerting if requested.
}

// ==== Main body ====
var customMapFormat = {
    name: yaz.description,
    extension: yaz.fileExtension,

    write: function(map, fileName) {
        var hdr = [];	// Used to collect the header data
        var mmd = [];	// Used to collect the main map data
		var mapProperties = map.properties();

		console.info("Exporting to", fileName);

		// === Part 1: Gather and validate all data ===
		for (let i = 0; i < map.layerCount; ++i) {
			let layer = map.layerAt(i);
			


			if (layer.isTileLayer && layer.name == yaz.layerNames.MAINMAP ) {
				/* File header */
				console.info( "Found:", layer.name, ". Layer: "+i );
				let stageTheme 	= yaz.clamp( parseInt( mapProperties[ yaz.attrNames.THEME ] ), 0, 3 );
				let stageSong	= yaz.clamp( parseInt( mapProperties[ yaz.attrNames.SONG ] ), 0, 6 );
				let stageAuthor	= mapProperties[ yaz.attrNames.AUTHOR ].substring(0, 3);
				let stageTitle	= mapProperties[ yaz.attrNames.TITLE ].substring(0, 19);

				// Build the header with the data we collected
				hdr = yaz.asciiEncoder ( yaz.magicID );		// Magic unique ID
				hdr.push( stageTheme );
				hdr.push( stageSong );
				hdr = hdr.concat( yaz.asciiEncoder( stageAuthor.padEnd(3,'\0') ) );
				hdr = hdr.concat( yaz.asciiEncoder( stageTitle.padEnd(19,'\xFF') ) );
				hdr.push( 255 );	// Header end

				/* Actual Map */
				console.info( "Found map. Layer: "+i );
				for( let y = 0; y < layer.height; ++y ){
					for( let x = 0; x < layer.width; ++x ){
						let tileID = (layer.cellAt(x, y).tileId <= yaz.maxTile )?layer.cellAt(x, y).tileId:0;
						mmd.push( tileID );	  
					}  
				}
			}
		}

		if( hdr.length < yaz.headerLength ) {
			return ("Yazzie Header layer error!");
		}
		if( mmd.length < 32*23 ) {
			return ("Yazzie Tile Map layer error!");
		}

		// === Part 2: Save the file ===
		var file = new BinaryFile(fileName, BinaryFile.WriteOnly);
		var newhdr = new Uint8Array(hdr).buffer;
		var newmap = new Uint8Array(mmd).buffer;

		file.write( newhdr );
		file.write( newmap );
		file.commit();
		console.log("Export", yaz.fileExtension, "format completed.");
    },
	read: function( fileName ) {
		const dirName = fileName.match(/.*[\/\\]/);		// Since path.basename isn't available, I had to reinvent the wheel
		var file = new BinaryFile(fileName, BinaryFile.ReadOnly);
		var fileData = file.readAll();
		var fileBytes = new Uint8Array( fileData );
		file.close();

		// Check the header signature, if necessary
		if( yaz.magicID.length > 0 ) {
			var magicID	= String.fromCharCode( ...fileBytes.slice(  0, yaz.magicID.length )).replace(/\x00/g, '');
			if( magicID != yaz.magicID ){
				tiled.error("Invalid yaz file.");
				//tiled.trigger("ViewIssues");		// Wishlist
				return;
			}
		}

		// Get the header attributes
		var theme	= fileBytes[7]
		var song	= fileBytes[8];
		var author	= String.fromCharCode( ...fileBytes.slice(  9, 12 )).replace(/\x00/g, '');
		var title	= String.fromCharCode( ...fileBytes.slice( 12, 32 )).replace(/\xFF/g, '');

		// Create the map
		var map = new TileMap();
        map.setSize( 32, 23);
        map.setTileSize( 8, 8);
        map.orientation = TileMap.Orthogonal;
		map.backgroundColor = "#1b1b1b";
		map.layerDataFormat = "CSV";
		map.setProperty( yaz.attrNames.THEME, theme);
		map.setProperty( yaz.attrNames.SONG, song);
		map.setProperty( yaz.attrNames.AUTHOR, author);
		map.setProperty( yaz.attrNames.TITLE, title);


		// Attach a Tile set
		var tileset =  tiled.open( dirName+yaz.tileSetName );
        if( tileset && tileset.isTileset ){
            map.addTileset( tileset );
			// tiled.close( tileset );
		} else tiled.error("Invalid Tile Set:", yaz.tileSetName );
		
		// Create the Tile layer
		var tileLayer 		= new TileLayer();
		tileLayer.width 	= map.width;
		tileLayer.height	= map.height;
		tileLayer.name		= yaz.layerNames.MAINMAP;
		var layerEdit		= tileLayer.edit();

		// Copy file data to the layer
		for (let y = 0; y < map.height; ++y) {
			for (let x = 0; x < map.width; ++x) {
				let tileID = fileBytes[yaz.headerLength+32*y+x];
				if( tileID <= yaz.maxTile ) 
					layerEdit.setTile( x, y, tileset.tile(tileID) );
			}
		}
		layerEdit.apply();
		map.addLayer( tileLayer );
		tiled.close( tileset );			

		if ( yaz.alertProperties )
			yaz.alertProperties = tiled.confirm( "File loaded.\r\n\r\nRemember to set the Author, Song, Theme and Title on Map→Map Properties\r\n\r\nRemind this again on this session?", yaz.description );

		return map;
	}
}

// ==== Register this plugin ====
tiled.registerMapFormat(yaz.displayName, customMapFormat)

